/*
 * Copyright 2012 Jesse Long.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package allinone;

import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.LinkedList;
import java.util.List;
import java.util.jar.Manifest;

public class Main
{
    private Main()
    {
    }

    private static String getMainClass(String urlBase)
            throws IOException
    {
        URL mainManifestURL = new URL(urlBase + "main/META-INF/MANIFEST.MF");
        InputStream in = mainManifestURL.openStream();
        try {
            return new Manifest(in).getMainAttributes().getValue("Main-Class");
        }finally{
            in.close();
        }
    }

    private static List<String> getDependencyPaths(URL dependencyListURL) throws IOException
    {
        List<String> dependencyPaths = new LinkedList<String>();
        BufferedReader r = new BufferedReader(new InputStreamReader(dependencyListURL.openStream(), Charset.forName("UTF-8")));
        try{
            String line;
            while ((line = r.readLine()) != null){
                line = line.trim();
                String[] parts = line.split(":");
                if (parts.length == 5){
                    dependencyPaths.add("lib/" + parts[0] + "." + parts[1] + "-" + parts[3] + "/");
                }
            }
        }finally{
            r.close();
        }

        return dependencyPaths;
    }

    public static void main(String[] args)
            throws FileNotFoundException, IOException, ClassNotFoundException, NoSuchMethodException, IllegalAccessException, IllegalArgumentException, InvocationTargetException
    {
        URL dependencyListURL = Main.class.getResource("allinone.txt");

        if (dependencyListURL == null){
            throw new FileNotFoundException("Missing resource: allinone.txt");
        }

        String dependencyListUrlString = dependencyListURL.toString();

        if (!dependencyListUrlString.endsWith("/allinone/allinone.txt")){
            throw new IllegalStateException("Unable to load main class from url: " + dependencyListUrlString);
        }

        String urlBase = dependencyListUrlString.substring(0, dependencyListUrlString.length() - "allinone/allinone.txt".length());

        String mainClassName = getMainClass(urlBase);
        
        if (mainClassName == null){
            throw new AssertionError("No main class specified");
        }

        List<String> dependencyPaths = getDependencyPaths(dependencyListURL);

        List<URL> urls = new LinkedList<URL>();
        urls.add(new URL(urlBase + "main/"));

        for (int i = 0; i < dependencyPaths.size(); i++){
            urls.add(new URL(urlBase + dependencyPaths.get(i)));
        }

        ClassLoader loader = new MultiURLClassLoader(Main.class.getClassLoader(), urls);

        System.setSecurityManager(null);
        Thread.currentThread().setContextClassLoader(loader);

        Class<?> mainClass = loader.loadClass(mainClassName);
        Method m = mainClass.getMethod("main", new String[]{}.getClass());

        if (!Modifier.isPublic(m.getModifiers()) || !Modifier.isStatic(m.getModifiers())){
            throw new IllegalStateException("No main() method in " + mainClassName);
        }

        m.invoke(null, new Object[]{args});
    }
}
